/*
*  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
*
*  This file is part of the Arnold emulator source code distribution.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include "memrange.h"
#include "cpc.h"

int MemoryRange_GetSize(const MemoryRange *pMemoryRange)
{
    if (pMemoryRange==NULL)
        return 0;
    return pMemoryRange->m_nLength;
}

BOOL MemoryRange_IsReadOnly(const MemoryRange *pMemoryRange)
{
    return pMemoryRange->m_bReadOnly;
}

// read little endian word
void MemoryRange_ReadWord(const MemoryRange *pMemoryRange, int nOffset, int *pData, int *pMask)
{
	int WordData = 0;
	int WordMask = 0;

	int ByteData = 0;
	int ByteMask = 0;
	MemoryRange_ReadByte(pMemoryRange, nOffset, &ByteData, &ByteMask);
	WordData |= (ByteData&0x0ff);
	WordMask |= (ByteMask&0x0ff);
	MemoryRange_ReadByte(pMemoryRange, nOffset+1, &ByteData, &ByteMask);
	WordData |= (ByteData<<8);
	WordMask |= (ByteMask<<8);

	*pData = WordData;
	*pMask = WordMask;
}


void MemoryRange_ReadByte(const MemoryRange *pMemoryRange, int nOffset, int *Data, int *Mask)
{
	int nReadOffset;

	*Mask = 0;
	*Data = 0;

    if (pMemoryRange==NULL)
        return;

	if (pMemoryRange->m_bCPU)
	{
		// what is mapped?
		int nMask = CPU_RD_MEM_MASK(nOffset) & 0x0ff;
		if (Mask == 0)
		{
			*Data = 0;
		}
		else
		{
			// mapped, return read data
			*Data = (CPU_RD_MEM(nOffset) & 0x0ff);
		}
		*Mask = nMask;
		return;
	}

	if (pMemoryRange->pBase == NULL)
	{
		return;
	}
	nReadOffset = (nOffset%pMemoryRange->m_nLength);
	if (pMemoryRange->pMappedMask && pMemoryRange->pMappedMask[nReadOffset] == 0)
	{
		*Data = 0;
		return;
	}

	*Data = (pMemoryRange->pBase[(nOffset%pMemoryRange->m_nLength)] & 0x0ff);
	*Mask = 0x0ff;
}

void MemoryRange_WriteByte(const MemoryRange *pMemoryRange, int nOffset, int nByte)
{
	// TODO: Handle unmapped?
    if (pMemoryRange==NULL)
        return;

    if (pMemoryRange->m_bReadOnly)
        return;

    if (pMemoryRange->m_bCPU)
    {
        CPU_WR_MEM(nOffset, nByte);
        return;
    }

    if (pMemoryRange->pBase==NULL)
        return;

    pMemoryRange->pBase[(nOffset%pMemoryRange->m_nLength)] = nByte;
}


void MemoryRange_WriteWord(const MemoryRange *pMemoryRange, int nOffset, int nWord)
{
	MemoryRange_WriteByte(pMemoryRange, nOffset, nWord & 0x0ff);
	MemoryRange_WriteByte(pMemoryRange, nOffset+1, (nWord>>8) & 0x0ff);
}
